// --- Constants ---
const API_URL = 'http://192.168.0.182';
const POLL_INTERVAL_MS = 300;
const MIN_DISTANCE_CM = 10;
const MAX_DISTANCE_CM = 100;
const BLIP_FADE_DURATION_MS = 3000;

// --- DOM Elements ---
const canvas = document.getElementById('radarCanvas');
const container = document.getElementById('container');
const connectionStatusEl = document.getElementById('connection-status');
const connectionTextEl = document.getElementById('connection-text');
const iconOnlineEl = document.getElementById('icon-online');
const iconOfflineEl = document.getElementById('icon-offline');
const pollRateEl = document.getElementById('poll-rate');
const latencyEl = document.getElementById('latency');
const distanceEl = document.getElementById('distance');
const positionEl = document.getElementById('position');
const proximityAlertEl = document.getElementById('proximity-alert');
const ctx = canvas.getContext('2d');

// --- State ---
let blips = [];
let lastPosition = 90;
let isConnected = false;
let lastUpdate = Date.now();
let lastDistance = null;


let dimensions = {
    width: container.clientWidth,
    height: container.clientHeight
};

function setupCanvas() {
    dimensions.width = container.clientWidth;
    dimensions.height = container.clientHeight;
    if (canvas.width !== dimensions.width || canvas.height !== dimensions.height) {
        canvas.width = dimensions.width;
        canvas.height = dimensions.height;
    }
}

// --- UI Updates ---
function updateConnectionStatus(connected) {
    isConnected = connected;
    if (connected) {
        connectionStatusEl.classList.remove('border-red-800', 'bg-red-900/20');
        connectionStatusEl.classList.add('border-green-800', 'bg-green-900/20');
        connectionTextEl.classList.remove('text-red-500');
        connectionTextEl.classList.add('text-green-400');
        connectionTextEl.innerText = 'SYSTEM ONLINE';
        iconOnlineEl.classList.remove('hidden'); 
        iconOfflineEl.classList.add('hidden');    
    } else {
        connectionStatusEl.classList.remove('border-green-800', 'bg-green-900/20');
        connectionStatusEl.classList.add('border-red-800', 'bg-red-900/20');
        connectionTextEl.classList.remove('text-green-400');
        connectionTextEl.classList.add('text-red-500');
        connectionTextEl.innerText = 'CONNECTION LOST';
        iconOnlineEl.classList.add('hidden');     
        iconOfflineEl.classList.remove('hidden');
    }
}

// --- API Fetching ---
async function fetchRadarData() {
    const startTime = Date.now();
    const controller = new AbortController();
    const timeoutId = setTimeout(() => controller.abort(), 1000);
    
    try {
        const response = await fetch(API_URL, { signal: controller.signal });
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        const newData = await response.json();
        handleNewData(newData);
        if (!isConnected) updateConnectionStatus(true);
        lastUpdate = Date.now();
    } catch (err) {
        updateConnectionStatus(false);
        console.warn('Failed to fetch radar data:', err);
    } finally {
        clearTimeout(timeoutId);
        const endTime = Date.now();
        const latency = endTime - startTime;
        latencyEl.innerText = `LATENCY: ${latency}ms`;
    }
}

function handleNewData(data) {
    if (!data) return;

    lastPosition = data.rot;
    if (data.dis != 0) {
        lastDistance = data.dis;
    }
    positionEl.innerText = data.rot ?? '--';

    if (data.dis >= MIN_DISTANCE_CM && data.dis <= MAX_DISTANCE_CM) {
        distanceEl.innerText = data.dis;
        const newBlip = {
            id: `${Date.now()}-${Math.random()}`,
            distance: data.dis,
            angle: data.rot,
            timestamp: Date.now(),
        };
        blips.push(newBlip);
    } else {
        distanceEl.innerText = lastDistance;
    }

    if (data.dis < 30 && data.dis >= MIN_DISTANCE_CM) {
        proximityAlertEl.classList.remove('opacity-0');
    } else {
        proximityAlertEl.classList.add('opacity-0');
    }
}

// --- Drawing ---
function drawGrid(ctx, centerX, centerY, maxRadius) {
    ctx.lineWidth = 1;
    ctx.strokeStyle = '#064e3b'; // Dark green-900

    const steps = 4;
    for (let i = 1; i <= steps; i++) {
        const r = (maxRadius / steps) * i;
        ctx.beginPath();
        ctx.arc(centerX, centerY, r, Math.PI, 0);
        ctx.stroke();

        ctx.fillStyle = '#065f46';
        ctx.font = '10px monospace';
        ctx.textAlign = 'right';
        ctx.fillText(`${(MAX_DISTANCE_CM / steps) * i}cm`, centerX + r - 5, centerY - 5);
    }

    const angleSteps = 6;
    for (let i = 0; i <= angleSteps; i++) {
        const angleDeg = (180 / angleSteps) * i;
        const angleRad = (angleDeg * Math.PI) / 180;
        const x = centerX + Math.cos(-angleRad) * maxRadius;
        const y = centerY + Math.sin(-angleRad) * maxRadius;

        ctx.beginPath();
        ctx.moveTo(centerX, centerY);
        ctx.lineTo(x, y);
        ctx.stroke();

        ctx.fillText(`${angleDeg}°`, x, y - 5);
    }
}

function drawBlips(ctx, centerX, centerY, maxRadius) {
    const now = Date.now();
    blips = blips.filter(b => now - b.timestamp < BLIP_FADE_DURATION_MS);

    blips.forEach(blip => {
        const age = now - blip.timestamp;
        const lifePct = 1 - (age / BLIP_FADE_DURATION_MS);

        const rPx = (blip.distance / MAX_DISTANCE_CM) * maxRadius;
        const rad = (blip.angle * Math.PI) / 180;

        const bx = centerX + rPx * Math.cos(-rad);
        const by = centerY + rPx * Math.sin(-rad);

        ctx.beginPath();
        ctx.fillStyle = `rgba(34, 197, 94, ${lifePct})`; // Green-500
        ctx.shadowBlur = 10;
        ctx.shadowColor = '#4ade80';
        ctx.arc(bx, by, 12, 0, 2 * Math.PI);
        ctx.fill();
        ctx.shadowBlur = 0;

        const pulseRadius = 6 + (age / 30);
        ctx.beginPath();
        ctx.strokeStyle = `rgba(74, 222, 128, ${lifePct * 0.5})`;
        ctx.lineWidth = 1;
        ctx.arc(bx, by, pulseRadius, 0, 2 * Math.PI);
        ctx.stroke();
    });
}

function drawSweep(ctx, centerX, centerY, maxRadius) {
    const sweepRad = (lastPosition * Math.PI) / 180;
    const tipX = centerX + maxRadius * Math.cos(-sweepRad);
    const tipY = centerY + maxRadius * Math.sin(-sweepRad);

    ctx.beginPath();
    ctx.moveTo(centerX, centerY);
    ctx.lineTo(tipX, tipY);
    ctx.strokeStyle = isConnected ? '#10b981' : '#ef4444';
    ctx.lineWidth = 4;
    ctx.shadowColor = isConnected ? '#10b981' : '#ef4444';
    ctx.shadowBlur = 15;
    ctx.stroke();
    ctx.shadowBlur = 0;
}

// --- Animation Loop ---
function animate() {
    setupCanvas();
    const { width, height } = dimensions;
    const centerX = width / 2;
    const centerY = height - (height * 0.1);
    const maxRadius = Math.min(width / 2, height * 0.85);

    ctx.fillStyle = 'rgba(0, 0, 0, 1)';
    ctx.fillRect(0, 0, width, height);

    drawGrid(ctx, centerX, centerY, maxRadius);
    drawBlips(ctx, centerX, centerY, maxRadius);
    drawSweep(ctx, centerX, centerY, maxRadius);

    requestAnimationFrame(animate);
}

// --- Initialization ---
function init() {
    window.addEventListener('resize', setupCanvas);
    pollRateEl.innerText = `POLL_RATE: ${POLL_INTERVAL_MS}ms`;

    // Start polling
    setInterval(fetchRadarData, POLL_INTERVAL_MS);
    fetchRadarData(); // Initial fetch

    // Start animation
    animate();
}

init();